/*
################################################################################
#                                                                              #
#    Ambient Lights Controller Firmware                                        #
#                                                                              #
#    Copyright 2012 Robert Mai                                                 #
#                                                                              #
#    Contact: robert@hsapps.com                                                #
#    URL:     http://www.hsapps.com                                            #
#                                                                              #
#                                                                              #
#    Ambient Lights Controller Firmware is free software: you can              #
#    redistribute it and/or modify it under the terms of the GNU General       #
#    Public License as published by the Free Software Foundation, either       #
#    version 3 of the License, or (at your option) any later version.          #
#                                                                              #
#    Ambient Lights Controller Firmware is distributed in the hope that        #
#    it will be useful, but WITHOUT ANY WARRANTY; without even the implied     #
#    warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.          #
#    See the GNU General Public License for more details.                      #
#                                                                              #
#    You should have received a copy of the GNU General Public License         #
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.     #
#                                                                              #
################################################################################
*/

#include "EncoderButton.h"
#include "Arduino.h"

EncoderButton::EncoderButton()
{
    evtButton = false;
    evtButtonClick = false;
    evtButtonLongClick = false;
    evtButtonDownRight = false;
    evtButtonDownLeft = false;
    evtButtonRight = false;
    evtButtonLeft = false;
}

void EncoderButton::setup(int pinA, int pinB, int pinBtn, unsigned long longClickTime)
{
  analog = false;
  pinMode(pinA,INPUT);
  pinMode(pinB,INPUT);  
  pinMode(pinBtn,INPUT);  
  doSetup(pinA, pinB, pinBtn, longClickTime);
}

void EncoderButton::setupAnalog(int pinA, int pinB, int pinBtn, unsigned long longClickTime)
{
  analog = true;
  doSetup(pinA, pinB, pinBtn, longClickTime);
}

void EncoderButton::doSetup(int pinA, int pinB, int pinBtn, unsigned long longClickTime)
{
  PIN_BTN_A = pinA;
  PIN_BTN_B = pinB;
  PIN_BTN_BTN = pinBtn;
  BUTTON_LONG_TIME = longClickTime;
  
  valA = doRead(PIN_BTN_A);
  valB = doRead(PIN_BTN_B);
  valBTN = doRead(PIN_BTN_BTN);
  btnDownTime = millis();
  btnDirection = BUTTON_RIGHT;
  
  previousMillisA = 0;
  previousMillisB = 0;
  previousMillisBtn = 0;
}

int EncoderButton::doRead(int pin)
{
  if (analog)
  {
    if (analogRead(pin) > 512)
      return HIGH;
    else
      return LOW;
  }
  else
  {
    return digitalRead(pin);
  }
}

void EncoderButton::process()
{
  evtButtonClick = false;
  evtButtonLongClick = false;
  evtButtonDownRight = false;
  evtButtonDownLeft = false;
  evtButtonRight = false;
  evtButtonLeft = false;
  
  unsigned long now = millis();
  newBTN = debounce(doRead(PIN_BTN_BTN), valBTN, now, &previousMillisBtn);
  newA = debounce(doRead(PIN_BTN_A), valA, now, &previousMillisA);
  newB = debounce(doRead(PIN_BTN_B), valB, now, &previousMillisB);
  
  if (newBTN != valBTN)
  {
    valBTN = newBTN;
    if (valBTN == BUTTON_UP && btnClickInterfered == false)
    {
      if ( (millis()-btnDownTime) > BUTTON_LONG_TIME )
      {
        evtButtonLongClick = true;
      }
      else
      {
        evtButtonClick = true;
      }
    }
    else if (valBTN == BUTTON_DOWN)
    {
      btnDownTime = millis();
      btnClickInterfered = false;
    }
  }
  
  if (newA != valA || newB != valB)
  {
    if ( ((newA<valA) && (newB == 1))
       || ((newA>valA) && (newB == 0)) )
       {
         btnDirection = BUTTON_LEFT;
       }
    if ( ((newB<valB) && (newA == 1))
       || ((newB>valB) && (newA == 0)) )
       {
         btnDirection = BUTTON_RIGHT;
       }
    if (newA == newB)
    {
      btnClickInterfered = true;
      evtButtonDownRight = (valBTN == BUTTON_DOWN) && (btnDirection == BUTTON_RIGHT);
      evtButtonDownLeft = (valBTN == BUTTON_DOWN) && (btnDirection == BUTTON_LEFT);
      evtButtonRight = (valBTN == BUTTON_UP) && (btnDirection == BUTTON_RIGHT);
      evtButtonLeft = (valBTN == BUTTON_UP) && (btnDirection == BUTTON_LEFT);
    }
    valA = newA;
    valB = newB;
  }
  evtButton = evtButtonClick || evtButtonLongClick || evtButtonDownRight || evtButtonDownLeft || evtButtonRight || evtButtonLeft;
}

int EncoderButton::debounce(int newVal, int oldVal, unsigned long currentMillis, unsigned long *previousMillis)
{
  if ( (newVal != oldVal) && (currentMillis - *previousMillis >= DEBOUNCE_TIME_MS) )
  {
    *previousMillis = currentMillis;
    return newVal;
  }
  else
  {
    return oldVal;
  }
}
