/*
################################################################################
#                                                                              #
#    Ambient Lights Controller Firmware                                        #
#                                                                              #
#    Copyright 2012 Robert Mai                                                 #
#                                                                              #
#    Contact: robert@hsapps.com                                                #
#    URL:     http://www.hsapps.com                                            #
#                                                                              #
#                                                                              #
#    Ambient Lights Controller Firmware is free software: you can              #
#    redistribute it and/or modify it under the terms of the GNU General       #
#    Public License as published by the Free Software Foundation, either       #
#    version 3 of the License, or (at your option) any later version.          #
#                                                                              #
#    Ambient Lights Controller Firmware is distributed in the hope that        #
#    it will be useful, but WITHOUT ANY WARRANTY; without even the implied     #
#    warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.          #
#    See the GNU General Public License for more details.                      #
#                                                                              #
#    You should have received a copy of the GNU General Public License         #
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.     #
#                                                                              #
################################################################################
*/

#include "LEDController.h"
#include "Arduino.h"

LEDController::LEDController()
{
    r = 0;
    g = 0;
    b = 0;
    rgbToHSV();
    forceOn = false;
}

void LEDController::setup(int vR, int vG, int vB)
{
    pinR = vR;
    pinG = vG;
    pinB = vB;
    pinMode(pinR,OUTPUT);
    pinMode(pinG,OUTPUT);
    pinMode(pinB,OUTPUT);
}

void LEDController::process()
{
  if (forceOn == true)
  {
    if (millis() > forceTargetTime)
    {
      forceOn = false;
      refreshLEDs();
    }
  }
}

void LEDController::setHue(int aHue)
{
    if (aHue < HUE_MIN)
    {
      setHue(HUE_MAX - (HUE_MIN-aHue) + 1);
    }
    else if (aHue > HUE_MAX)
    {
      setHue(HUE_MIN + (aHue-HUE_MAX) - 1);
    }
    else
    {
      hue = aHue;
      hsvToRGB();
      refreshLEDs();
    }
}

void LEDController::setSaturation(int aSaturation)
{
    if (aSaturation < SATURATION_MIN)
    {
      aSaturation = SATURATION_MIN;
    }
    else if (aSaturation > SATURATION_MAX)
    {
      aSaturation = SATURATION_MAX;
    }

    saturation = aSaturation;
    hsvToRGB();
    refreshLEDs();
}

void LEDController::setValue(int aValue)
{
    if (aValue < VALUE_MIN)
    {
      aValue = VALUE_MIN;
    }
    else if (aValue > VALUE_MAX)
    {
      aValue = VALUE_MAX;
    }

    value = aValue;
    hsvToRGB();
    refreshLEDs();
}

int LEDController::getR()
{
    return r;
}

int LEDController::getG()
{
    return g;
}

int LEDController::getB()
{
    return b;
}

int LEDController::getHue()
{
    return hue;
}

int LEDController::getSaturation()
{
    return saturation;
}

int LEDController::getValue()
{
    return value;
}

void LEDController::setRGB(int vR, int vG, int vB)
{
    r = vR;
    g = vG;
    b = vB;
    rgbToHSV();
    refreshLEDs();
}
void LEDController::setRGBHelper(float vR, float vG, float vB)
{
    r = (int)(vR*255.0);
    g = (int)(vG*255.0);
    b = (int)(vB*255.0);
}
void LEDController::hsvToRGB()
{
    float H = ((float)hue); // 0..360
    float S = ((float)saturation)/255.0; //now 0..1
    float V = ((float)value)/255.0; //now 0..1
    
    float hi = (float)((int)(H/60.0));
    float f = H/60.0 - hi;
    
    float p = V*(1.0-S);
    float q = V*(1.0-S*f);
    float t = V*(1.0-S*(1.0-f));
    
    if (hi < 0.5)
        setRGBHelper(V,t,p);
    else if (hi < 1.5)
        setRGBHelper(q,V,p);
    else if (hi < 2.5)
        setRGBHelper(p,V,t);
    else if (hi < 3.5)
        setRGBHelper(p,q,V);
    else if (hi < 4.5)
        setRGBHelper(t,p,V);
    else if (hi < 5.5)
        setRGBHelper(V,p,q);
    else
        setRGBHelper(V,t,p);
}

void LEDController::rgbToHSV()
{
    float R = ((float)r) / 255.0;
    float G = ((float)g) / 255.0;
    float B = ((float)b) / 255.0;
    float MAX = R;
    if (G > MAX)
        MAX = G;
    if (B > MAX)
        MAX = B;
    float MIN = R;
    if (G < MIN)
        MIN = G;
    if (B < MIN)
        MIN = B;
    
    float H = 0.0;
    if (MAX == MIN)
        H = 0.0;
    else if (MAX == R)
        H = 60.0 * (G-B) / (MAX - MIN);
    else if (MAX == G)
        H = 60.0 * (2.0 + (B-R) / (MAX - MIN) );
    else
        H = 60.0 * (4.0 + (R-G) / (MAX - MIN) );
    
    float S = 0.0;
    if (MAX > 0.0)
        S = (MAX-MIN)/MAX;
    
    hue = (int)(H);
    saturation = (int)(S * 255.0);
    value = (int)(MAX * 255.0);
}

void LEDController::refreshLEDs()
{
    if (forceOn == true)
    {
      analogWrite(pinR,forceR);
      analogWrite(pinG,forceG);
      analogWrite(pinB,forceB);
    }
    else
    {
      analogWrite(pinR,r);
      analogWrite(pinG,g);
      analogWrite(pinB,b);
    }
}

void LEDController::forceRGB(int vR, int vG, int vB, unsigned long duration)
{
  forceR = vR;
  forceG = vG;
  forceB = vB;
  
  forceTargetTime = millis() + duration;
  forceOn = true;
  refreshLEDs();
}

void LEDController::forceHSV(int vH, int vS, int vV, unsigned long duration)
{
  int tH = hue;
  int tS = saturation;
  int tV = value;
  
  int tR = r;
  int tG = g;
  int tB = b;
  
  setHSV(vH,vS,vV);
  hsvToRGB();
  
  forceR = r;
  forceG = g;
  forceB = b;
  
  hue = tH;
  saturation = tS;
  value = tV;
  
  r = tR;
  g = tG;
  b = tB;
  
  forceTargetTime = millis() + duration;
  forceOn = true;
  refreshLEDs();
}
    
void LEDController::setHSV(int vH, int vS, int vV)
{
  if (vH < HUE_MIN)
    vH = HUE_MAX - (HUE_MIN-vH) + 1;
  else if (vH > HUE_MAX)
    vH = HUE_MIN + (vH-HUE_MAX) - 1;
  
  if (vS < SATURATION_MIN)
    vS = SATURATION_MIN;
  else if (vS > SATURATION_MAX)
    vS = SATURATION_MAX;

  if (vV < VALUE_MIN)
    vV = VALUE_MIN;
  else if (vV > VALUE_MAX)
    vV = VALUE_MAX;
  
  hue = vH;
  saturation = vS;
  value = vV;
  hsvToRGB();
  refreshLEDs();
}

void LEDController::synchronize(LEDController* source)
{
    hue = source->hue;
    saturation = source->saturation;
    value = source->value;
    
    r = source->r;
    g = source->g;
    b = source->b;
    
    forceR = source->forceR;
    forceG = source->forceG;
    forceB = source->forceB;
    forceOn = source->forceOn;
    forceTargetTime = source->forceTargetTime;
    
    refreshLEDs();
}
