/*
################################################################################
#                                                                              #
#    Ambient Lights Controller Firmware                                        #
#                                                                              #
#    Copyright 2012 Robert Mai                                                 #
#                                                                              #
#    Contact: robert@hsapps.com                                                #
#    URL:     http://www.hsapps.com                                            #
#                                                                              #
#                                                                              #
#    Ambient Lights Controller Firmware is free software: you can              #
#    redistribute it and/or modify it under the terms of the GNU General       #
#    Public License as published by the Free Software Foundation, either       #
#    version 3 of the License, or (at your option) any later version.          #
#                                                                              #
#    Ambient Lights Controller Firmware is distributed in the hope that        #
#    it will be useful, but WITHOUT ANY WARRANTY; without even the implied     #
#    warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.          #
#    See the GNU General Public License for more details.                      #
#                                                                              #
#    You should have received a copy of the GNU General Public License         #
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.     #
#                                                                              #
################################################################################
*/

#include "StateSaver.h"
#include "Arduino.h"

#include <EEPROM.h> //YOU MUST ALSO INCLUDE THIS IN THE MAIN PDE FILE BEFORE INCLUDING StateSaver.h!


StateSaver::StateSaver()
{
  prevH1 = 0;
  prevS1 = 0;
  prevV1 = 0;
  sameSince1 = millis();
  stored1 = false;
  prevH2 = 0;
  prevS2 = 0;
  prevV2 = 0;
  sameSince2 = millis();
  stored2 = false;
  startTokenAddress = 0;
  
  storedH1 = 0;
  storedS1 = 0;
  storedV1 = 0;
  storedH2 = 0;
  storedS2 = 0;
  storedV2 = 0;
  
  fading = false;
}

void StateSaver::setup(LEDController* vLed1,LEDController* vLed2)
{
  led1 = vLed1;
  led2 = vLed2;
}

boolean StateSaver::processChannel(LEDController* led, int* prevH, int* prevS, int* prevV, unsigned long* sameSince, boolean* stored)
{
  //returns true if this channel needs to be stored
    int newH = led->getHue();
    int newS = led->getSaturation();
    int newV = led->getValue();
    
    if (newH != *prevH || newS != *prevS || newV != *prevV)
    { //color changed: remember it
      *prevH = newH;
      *prevS = newS;
      *prevV = newV;
      *sameSince = millis();
      *stored = false;
    }
    else
    { //still the same
      if ((*stored == false) && ( (millis() - *sameSince) > SAVE_INTERVAL_MS ))
      { //save it!
        *stored = true;
        return true;
      }
    }
    return false;  
}


void StateSaver::process()
{
  if (fading == true)
  { //still fading: don't save anything. Just fade
    unsigned long deltaTime = millis() - fadeStart; //time since start of fading
    if (deltaTime >= fadeDelayTarget)
    {
      led1->setValue((int) fadeVTarget1);
      led2->setValue((int) fadeVTarget2);
      fading = false;
    }
    else
    {
      int fadeValue = (int) ( ((float) deltaTime / fadeDelayTargetF) * fadeVTarget1 );
      led1->setValue(fadeValue);
      fadeValue = (int) ( ((float) deltaTime / fadeDelayTargetF) * fadeVTarget2 );
      led2->setValue(fadeValue);
    }
  }
  else
  { //not fading: business as usual
    boolean doStore1 = processChannel(led1, &prevH1, &prevS1, &prevV1, &sameSince1, &stored1);
    boolean doStore2 = processChannel(led2, &prevH2, &prevS2, &prevV2, &sameSince2, &stored2);
    if (doStore1 || doStore2)
    {
      writeState(doStore1, doStore2);
    }
  }
}

void StateSaver::readState()
{
  readState(0);
}
void StateSaver::readState(unsigned long fadeDelay)
{
  if (EEPROM.read(MIN_ADDR) == TOKEN_START)
  { //valid, look for next TOKEN_START
    int addr = MIN_ADDR + 1;
    while (addr <= MAX_ADDR - 6 && EEPROM.read(addr) == TOKEN_SKIP)
    {
      addr++;
    }
    //now addr should point to second TOKEN_START
    
    //check if it's really TOKEN_START
    if (EEPROM.read(addr) == TOKEN_START)
    { //yes: read H, S and V
      startTokenAddress = addr;
      int h1 = EEPROM.read(addr+1);
      int s1 = EEPROM.read(addr+2);
      int v1 = EEPROM.read(addr+3);
      
      int h2 = EEPROM.read(addr+4);
      int s2 = EEPROM.read(addr+5);
      int v2 = EEPROM.read(addr+6);
      
      //do we want to fade?
      if (fadeDelay > 0)
      {
        led1->setHSV(h1,s1,0); //start value is 0
        led2->setHSV(h2,s2,0); //start value is 0
        fadeStart = millis();
        fadeDelayTarget = fadeDelay;
        fadeDelayTargetF = (float) fadeDelayTarget;
        fading = true;
        fadeVTarget1 = v1;
        fadeVTarget2 = v2;
      }
      else
      {
        led1->setHSV(h1,s1,v1);
        led2->setHSV(h2,s2,v2);
        fading = false;
      }
      prevH1 = h1;
      prevS1 = s1;
      prevV1 = v1;
      stored1 = true; //prevent it from being stored again
      prevH2 = h2;
      prevS2 = s2;
      prevV2 = v2;
      stored2 = true; //prevent it from being stored again
    }
  }
  else
  {
    //invalid, no data saved, cancel
  }
}

void StateSaver::writeState(boolean writeChannel1, boolean writeChannel2) //readState() must have been called before if you want to save to another address this time.
{
  if (startTokenAddress == 0)
  { //unknown, check it
    if (EEPROM.read(MIN_ADDR) != TOKEN_START)
    { //no data has been saved before. Write TOKEN_START for the first time!
      EEPROM.write(MIN_ADDR, TOKEN_START);
      EEPROM.write(MIN_ADDR+1, TOKEN_START);
      startTokenAddress = MIN_ADDR+1;
    }
    else
    { //find out where second TOKEN_START is written:
      int addr = MIN_ADDR + 1;
      while (addr <= MAX_ADDR - 6 && EEPROM.read(addr) == TOKEN_SKIP)
      {
        addr++;
      }
      //now addr should point to second TOKEN_START
      
      //check if it's really TOKEN_START
      if (EEPROM.read(addr) == TOKEN_START)
      {
        startTokenAddress = addr;
      }
      else
      { //should never happen! Fallback:
        EEPROM.write(MIN_ADDR, TOKEN_START);
        EEPROM.write(MIN_ADDR+1, TOKEN_START);
        startTokenAddress = MIN_ADDR+1;
      }
      
    }
  }
  
  //check if we should start from the beginning:
  if (startTokenAddress+1 > MAX_ADDR-6)
  { // yes: reset start token position:
    startTokenAddress = MIN_ADDR + 1;
  }
  else
  { // no: write TOKEN_SKIP to current position and increase it
    EEPROM.write(startTokenAddress, TOKEN_SKIP);
    startTokenAddress++;
  }
  //write TOKEN_START to current start token address followed by R, G and B:
  EEPROM.write(startTokenAddress, TOKEN_START);
  if (writeChannel1)
  {
    EEPROM.write(startTokenAddress+1, byte(prevH1));
    EEPROM.write(startTokenAddress+2, byte(prevS1));
    EEPROM.write(startTokenAddress+3, byte(prevV1));
    storedH1 = prevH1;
    storedS1 = prevS1;
    storedV1 = prevV1;
  }
  else
  { //store old values
    EEPROM.write(startTokenAddress+1, byte(storedH1));
    EEPROM.write(startTokenAddress+2, byte(storedS1));
    EEPROM.write(startTokenAddress+3, byte(storedV1));
  }
  if (writeChannel2)
  {
    EEPROM.write(startTokenAddress+4, byte(prevH2));
    EEPROM.write(startTokenAddress+5, byte(prevS2));
    EEPROM.write(startTokenAddress+6, byte(prevV2));
    storedH2 = prevH2;
    storedS2 = prevS2;
    storedV2 = prevV2;
  }
  else //store old values
  {
    EEPROM.write(startTokenAddress+4, byte(storedH2));
    EEPROM.write(startTokenAddress+5, byte(storedS2));
    EEPROM.write(startTokenAddress+6, byte(storedV2));
  }
  
  
}
