/*
################################################################################
#                                                                              #
#    Ambient Lights Controller Firmware                                        #
#                                                                              #
#    Copyright 2012 Robert Mai                                                 #
#                                                                              #
#    Contact: robert@hsapps.com                                                #
#    URL:     http://www.hsapps.com                                            #
#                                                                              #
#                                                                              #
#    Ambient Lights Controller Firmware is free software: you can              #
#    redistribute it and/or modify it under the terms of the GNU General       #
#    Public License as published by the Free Software Foundation, either       #
#    version 3 of the License, or (at your option) any later version.          #
#                                                                              #
#    Ambient Lights Controller Firmware is distributed in the hope that        #
#    it will be useful, but WITHOUT ANY WARRANTY; without even the implied     #
#    warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.          #
#    See the GNU General Public License for more details.                      #
#                                                                              #
#    You should have received a copy of the GNU General Public License         #
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.     #
#                                                                              #
################################################################################
*/

#include "StateMachine.h"
#include "WProgram.h"

StateMachine::StateMachine()
{
    state = STATE_MANUAL_HUE;
    resetAutoSpeed();
}

void StateMachine::setup(LEDController* vLed, int vDelta, int vBigDelta)
{
    led = vLed;
    delta = vDelta;
    bigDelta = vBigDelta;
}

void StateMachine::resetAutoSpeed()
{
    autoNextStep = 0;
    autoStepSize = AUTO_STEP_SIZE;
    autoDelay = AUTO_DELAY;
}

void StateMachine::onEvent(byte event)
{
    //perform action according to current state:
    //Manual State:
    if (state != STATE_AUTO)
    {
        if (event == EVENT_LONG_CLICK)
        {
            state = STATE_AUTO;
            resetAutoSpeed(); //reset values set by serial interface
            return;
        }
        if (event == EVENT_CLICK)
        {
            if (state == STATE_MANUAL_MAX)
                state = STATE_MANUAL_MIN;
            else
                state++;
            return;
        }
        if (state == STATE_MANUAL_HUE)
        {
            if (event == EVENT_RIGHT)
                led->setHue(led->getHue()+delta);
            else if (event == EVENT_LEFT)
                led->setHue(led->getHue()-delta);
            else if (event == EVENT_DOWN_RIGHT)
                led->setHue(led->getHue()+bigDelta);
            else if (event == EVENT_DOWN_LEFT)
                led->setHue(led->getHue()-bigDelta);
        }
        if (state == STATE_MANUAL_SATURATION)
        {
            if (event == EVENT_RIGHT)
                led->setSaturation(led->getSaturation()+delta);
            else if (event == EVENT_LEFT)
                led->setSaturation(led->getSaturation()-delta);
            else if (event == EVENT_DOWN_RIGHT)
                led->setSaturation(led->getSaturation()+bigDelta);
            else if (event == EVENT_DOWN_LEFT)
                led->setSaturation(led->getSaturation()-bigDelta);
        }
        if (state == STATE_MANUAL_VALUE)
        {
            if (event == EVENT_RIGHT)
                led->setValue(led->getValue()+delta);
            else if (event == EVENT_LEFT)
                led->setValue(led->getValue()-delta);
            else if (event == EVENT_DOWN_RIGHT)
                led->setValue(led->getValue()+bigDelta);
            else if (event == EVENT_DOWN_LEFT)
                led->setValue(led->getValue()-bigDelta);
        }
    }
    else
    {//STATE_AUTO
        if (event == EVENT_LONG_CLICK)
        {
            state = STATE_MANUAL_HUE;
            return;
        }
        else if (event == EVENT_RIGHT) //increase speed
            increaseAutoSpeed(false);
        else if (event == EVENT_LEFT) //decrease speed
            decreaseAutoSpeed(false);
        else if (event == EVENT_DOWN_RIGHT) //increase speed rapidly
            increaseAutoSpeed(true);
        else if (event == EVENT_DOWN_LEFT) //decrease speed rapidly
            decreaseAutoSpeed(true);
    }
}

void StateMachine::increaseAutoSpeed(boolean largeStep)
{
  //currently no support for large step...
  if (autoDelay > 0)
      autoDelay--;
  else
      autoStepSize++;
}

void StateMachine::decreaseAutoSpeed(boolean largeStep)
{
  //currently no support for large step...
  if (autoStepSize > AUTO_STEP_SIZE)
      autoStepSize--;
  else
      autoDelay++;
}

void StateMachine::setState(byte vState)
{
    state = vState;
}

byte StateMachine::getState()
{
  return state;
}

void StateMachine::printState()
{
  if (state == STATE_MANUAL_HUE)
    Serial.print("STATE_MANUAL_HUE\n");

  if (state == STATE_MANUAL_SATURATION)
    Serial.print("STATE_MANUAL_SATURATION\n");
    
  if (state == STATE_MANUAL_VALUE)
    Serial.print("STATE_MANUAL_VALUE\n");
    
  if (state == STATE_AUTO)
    Serial.print("STATE_AUTO\n");
}

int StateMachine::getAutoStepSize()
{
  return autoStepSize;
}

int StateMachine::getAutoDelay()
{
  return autoDelay;
}


void StateMachine::process()
{
  if (state == STATE_AUTO)
  {
    unsigned long time_now = millis();
    if (time_now > autoNextStep)
    {
      led->setHue(led->getHue() + autoStepSize);
      led->refreshLEDs();
      autoNextStep = time_now + autoDelay;
    }
  }
}

void StateMachine::onSetRGB(int vR, int vG, int vB) //called by serial interface
{
  state = STATE_MANUAL_HUE;
  led->setRGB(vR,vG,vB);
}

void StateMachine::onSetHSV(int vH, int vS, int vV) //called by serial interface
{
  state = STATE_MANUAL_HUE;
  led->setHSV(vH,vS,vV);
}

void StateMachine::onSetAuto(boolean active) //called by serial interface
{
  if (active == true)
  {
    state = STATE_AUTO;
  }
  else
  {
    state = STATE_MANUAL_HUE;
  }
}

void StateMachine::onSetAutoValues(int stepSize, int stepDelay) //called by serial interface
{
  autoStepSize = stepSize;
  autoDelay = stepDelay;
}
