/*
################################################################################
#                                                                              #
#    Ambient Lights Controller Firmware                                        #
#                                                                              #
#    Copyright 2012 Robert Mai                                                 #
#                                                                              #
#    Contact: robert@hsapps.com                                                #
#    URL:     http://www.hsapps.com                                            #
#                                                                              #
#                                                                              #
#    Ambient Lights Controller Firmware is free software: you can              #
#    redistribute it and/or modify it under the terms of the GNU General       #
#    Public License as published by the Free Software Foundation, either       #
#    version 3 of the License, or (at your option) any later version.          #
#                                                                              #
#    Ambient Lights Controller Firmware is distributed in the hope that        #
#    it will be useful, but WITHOUT ANY WARRANTY; without even the implied     #
#    warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.          #
#    See the GNU General Public License for more details.                      #
#                                                                              #
#    You should have received a copy of the GNU General Public License         #
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.     #
#                                                                              #
################################################################################
*/

#include "StateSaver.h"
#include "WProgram.h"

#include <EEPROM.h> //YOU MUST ALSO INCLUDE THIS IN THE MAIN PDE FILE BEFORE INCLUDING StateSaver.h!


StateSaver::StateSaver()
{
  prevH = 0;
  prevS = 0;
  prevV = 0;
  sameSince = millis();
  stored = false;
  startTokenAddress = 0;
  
  fading = false;
}

void StateSaver::setup(LEDController* vLed)
{
  led = vLed;
}

void StateSaver::process()
{
  if (fading == true)
  { //still fading: don't save anything. Just fade
    unsigned long deltaTime = millis() - fadeStart; //time since start of fading
    if (deltaTime >= fadeDelayTarget)
    {
      led->setValue((int) fadeVTarget);
      fading = false;
    }
    else
    {
      int fadeValue = (int) ( ((float) deltaTime / fadeDelayTargetF) * fadeVTarget );
      led->setValue(fadeValue);
    }
  }
  else
  { //not fading: business as usual
    int newH = led->getHue();
    int newS = led->getSaturation();
    int newV = led->getValue();
    
    if (newH != prevH || newS != prevS || newV != prevV)
    { //color changed: remember it
      prevH = newH;
      prevS = newS;
      prevV = newV;
      sameSince = millis();
      stored = false;
    }
    else
    { //still the same
      if ((stored == false) && ( (millis() - sameSince) > SAVE_INTERVAL_MS ))
      { //save it!
        writeState();
        stored = true;
        //debug: flash for info
        //led->forceRGB(0, 0, 0, 250);
      }
    }
  }
}

void StateSaver::readState()
{
  readState(0);
}
void StateSaver::readState(unsigned long fadeDelay)
{
  if (EEPROM.read(MIN_ADDR) == TOKEN_START)
  { //valid, look for next TOKEN_START
    int addr = MIN_ADDR + 1;
    while (addr <= MAX_ADDR - 3 && EEPROM.read(addr) == TOKEN_SKIP)
    {
      addr++;
    }
    //now addr should point to second TOKEN_START
    
    //check if it's really TOKEN_START
    if (EEPROM.read(addr) == TOKEN_START)
    { //yes: read H, S and V
      startTokenAddress = addr;
      int h = EEPROM.read(addr+1);
      int s = EEPROM.read(addr+2);
      int v = EEPROM.read(addr+3);
      
      //do we want to fade?
      if (fadeDelay > 0)
      {
        led->setHSV(h,s,0); //start value is 0
        fadeStart = millis();
        fadeDelayTarget = fadeDelay;
        fadeDelayTargetF = (float) fadeDelayTarget;
        fading = true;
        fadeVTarget = v;
      }
      else
      {
        led->setHSV(h,s,v);
        fading = false;
      }
      prevH = h;
      prevS = s;
      prevV = v;
      stored = true; //prevent it from being stored again
    }
  }
  else
  {
    //invalid, no data saved, cancel
  }
}

void StateSaver::writeState() //readState() must have been called before if you want to save to another address this time.
{
  if (startTokenAddress == 0)
  { //unknown, check it
    if (EEPROM.read(MIN_ADDR) != TOKEN_START)
    { //no data has been saved before. Write TOKEN_START for the first time!
      EEPROM.write(MIN_ADDR, TOKEN_START);
      EEPROM.write(MIN_ADDR+1, TOKEN_START);
      startTokenAddress = MIN_ADDR+1;
    }
    else
    { //find out where second TOKEN_START is written:
      int addr = MIN_ADDR + 1;
      while (addr <= MAX_ADDR - 3 && EEPROM.read(addr) == TOKEN_SKIP)
      {
        addr++;
      }
      //now addr should point to second TOKEN_START
      
      //check if it's really TOKEN_START
      if (EEPROM.read(addr) == TOKEN_START)
      {
        startTokenAddress = addr;
      }
      else
      { //should never happen! Fallback:
        EEPROM.write(MIN_ADDR, TOKEN_START);
        EEPROM.write(MIN_ADDR+1, TOKEN_START);
        startTokenAddress = MIN_ADDR+1;
      }
      
    }
  }
  
  //check if we should start from the beginning:
  if (startTokenAddress+1 > MAX_ADDR-3)
  { // yes: reset start token position:
    startTokenAddress = MIN_ADDR + 1;
  }
  else
  { // no: write TOKEN_SKIP to current position and increase it
    EEPROM.write(startTokenAddress, TOKEN_SKIP);
    startTokenAddress++;
  }
  //write TOKEN_START to current start token address followed by R, G and B:
  EEPROM.write(startTokenAddress, TOKEN_START);
  EEPROM.write(startTokenAddress+1, byte(prevH));
  EEPROM.write(startTokenAddress+2, byte(prevS));
  EEPROM.write(startTokenAddress+3, byte(prevV));
  
  
  
}
