/*
################################################################################
#                                                                              #
#    Ambient Lights Controller Firmware                                        #
#                                                                              #
#    Copyright 2012 Robert Mai                                                 #
#                                                                              #
#    Contact: robert@hsapps.com                                                #
#    URL:     http://www.hsapps.com                                            #
#                                                                              #
#                                                                              #
#    Ambient Lights Controller Firmware is free software: you can              #
#    redistribute it and/or modify it under the terms of the GNU General       #
#    Public License as published by the Free Software Foundation, either       #
#    version 3 of the License, or (at your option) any later version.          #
#                                                                              #
#    Ambient Lights Controller Firmware is distributed in the hope that        #
#    it will be useful, but WITHOUT ANY WARRANTY; without even the implied     #
#    warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.          #
#    See the GNU General Public License for more details.                      #
#                                                                              #
#    You should have received a copy of the GNU General Public License         #
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.     #
#                                                                              #
################################################################################
*/

#include "SerialMessageInterface.h"
#include "Arduino.h"

#include <stdint.h>
#include <MyHomeMessageLayer.h>


#define LVL_SILENT 999
#define LVL_DEBUG 1
#define LVL_INFO 2

#define VERBOSE_LEVEL LVL_INFO


// MESSAGE ID DEFINITIONS:
#define AL_SET_COLOR_RGB               0x01
#define AL_SET_COLOR_HSV               0x02
#define AL_SET_CONTROLLER_MODE         0x03
#define AL_SET_AUTO_MODE_SPEED         0x04
#define AL_REQUEST_CONTROLLER_STATE    0x05
#define AL_REPORT_CONTROLLER_STATE     0x06
#define AL_SET_CONTINUOUS_STATE_UPDATE 0x07

//ENUM DEFINITIONS:
//CONTROLLER_MODE
#define MODE_MANUAL_HUE 0x00
#define MODE_MANUAL_SATURATION 0x01
#define MODE_MANUAL_VALUE 0x02
#define MODE_AUTO 0x03

SerialMessageInterface::SerialMessageInterface()
{
}

void putByte(int c)
{
  Serial.write(c);
}

void SerialMessageInterface::setup(int baudRate, StateMachine* vSm1, LEDController* vLed1, StateMachine* vSm2, LEDController* vLed2) //, ASKModulator* vAsk)
{
    reader.init(SERVICE_AMBIENT_LIGHTS); //only accept messages with this service_id
    writer.setPutByteFunction(&putByte);
    sm1 = vSm1;
    led1 = vLed1;
    sm2 = vSm2;
    led2 = vLed2;
    statusMsg = NULL;
    Serial.begin(baudRate);
}

void SerialMessageInterface::process()
{
  byte c = 0;
  if (Serial.available()) {
    c = Serial.read();
    Message* msg = reader.putByte(c);
    if (msg != NULL && msg->service_id == SERVICE_AMBIENT_LIGHTS) {
      //Serial.println("GOT A MESSAGE!");
      processMessage(msg);
    }
  }
  
  //check if we should send a state update:
  if ( (continuousUpdateCh1 || continuousUpdateCh2) && (millis() >= nextContinuousUpdate) )
  {
    nextContinuousUpdate = millis() + updateInterval; //set next time's millis-value
    if (continuousUpdateCh1)
    {
      sendStatusMessage(0,sm1,led1);
    }
    if (continuousUpdateCh2)
    {
      sendStatusMessage(1,sm2,led2);
    }
  }
  
}

void SerialMessageInterface::printMessage(Message* msg)
{
  //mlayer_printMessage(msg);
}

void SerialMessageInterface::processMessage(Message* msg)
{
  if (msg->message_id >= AL_SET_COLOR_RGB && msg->message_id <= AL_SET_CONTINUOUS_STATE_UPDATE)
  { //it's a message with APPLY_TO_CHANNEL signals in first byte:
    boolean applyToChan1 = msg->payload[0] & 0x01;
    boolean applyToChan2 = msg->payload[0] & 0x01<<1;
    
    if (msg->message_id == AL_SET_COLOR_RGB)
    {
      if (applyToChan1)
      {
        sm1->onSetRGB(msg->payload[1],msg->payload[2],msg->payload[3]);
      }
      if (applyToChan2)
      {
        sm2->onSetRGB(msg->payload[1],msg->payload[2],msg->payload[3]);
      }
    }
    if (msg->message_id == AL_SET_COLOR_HSV)
    {
      int hue = (msg->payload[0] >> 4);
      hue += ((int) msg->payload[1])<<4;
      if (applyToChan1)
      {
        sm1->onSetHSV(hue,msg->payload[2],msg->payload[3]);
      }
      if (applyToChan2)
      {
        sm2->onSetHSV(hue,msg->payload[2],msg->payload[3]);
      }
    }
    if (msg->message_id == AL_SET_CONTROLLER_MODE)
    {
      int mode = (msg->payload[0] & 0xF0) >> 4;
      int state = 0;
      if (mode == MODE_MANUAL_HUE)
        state = STATE_MANUAL_HUE;
      if (mode == MODE_MANUAL_SATURATION)
        state = STATE_MANUAL_SATURATION;
      if (mode == MODE_MANUAL_VALUE)
        state = STATE_MANUAL_VALUE;
      if (mode == MODE_AUTO)
        state = STATE_AUTO;

      if (applyToChan1)
      {
        sm1->setState(state);
      }
      if (applyToChan2)
      {
        sm2->setState(state);
      }
    }
    if (msg->message_id == AL_SET_AUTO_MODE_SPEED)
    {
      int changeStep = (msg->payload[0] >> 4);
      changeStep += ((int) msg->payload[1])<<4;
      int changeInterval = msg->payload[2];
      changeInterval += ((int)msg->payload[3])<<8;
      
      if (applyToChan1)
      {
        sm1->onSetAutoValues(changeStep,changeInterval);
      }
      if (applyToChan2)
      {
        sm2->onSetAutoValues(changeStep,changeInterval);
      }
    }
    if (msg->message_id == AL_REQUEST_CONTROLLER_STATE)
    {
      if (applyToChan1)
      {
        sendStatusMessage(0,sm1,led1);
      }
      if (applyToChan2)
      {
        sendStatusMessage(1,sm2,led2);
      }
    }
    if (msg->message_id == AL_SET_CONTINUOUS_STATE_UPDATE) {
      continuousUpdateCh1 = applyToChan1;
      continuousUpdateCh2 = applyToChan2;
      nextContinuousUpdate = millis(); //now!
      updateInterval = msg->payload[1];
      updateInterval += msg->payload[2] << 8;
    }
  }
  
}


void SerialMessageInterface::sendStatusMessage(int channelNumber, StateMachine* sm, LEDController* led)
{
  if (statusMsg == NULL)
  {
    statusMsg = mlayer_createMessage(8); //create Message object with buffersize 8
    statusMsg->payload_length = 8;
    statusMsg->service_id = SERVICE_AMBIENT_LIGHTS;
    statusMsg->message_id = AL_REPORT_CONTROLLER_STATE;
  }
  statusMsg->payload[0] = channelNumber & 0x3; //only two lower bits
  statusMsg->payload[0] |= (led->getHue() & 0x0F) << 4;
  statusMsg->payload[1] = (led->getHue() & 0xFF0) >> 4;
  statusMsg->payload[2] = led->getSaturation();
  statusMsg->payload[3] = led->getValue();
  switch (sm->getState())
  {
    case STATE_MANUAL_HUE:
      statusMsg->payload[4] = MODE_MANUAL_HUE;
      break;
    case STATE_MANUAL_SATURATION:
      statusMsg->payload[4] = MODE_MANUAL_SATURATION;
      break;
    case STATE_MANUAL_VALUE:
      statusMsg->payload[4] = MODE_MANUAL_VALUE;
      break;
    case STATE_AUTO:
      statusMsg->payload[4] = MODE_AUTO;
      break;
    default:
      statusMsg->payload[4] = 0x0F; //invalid!?!
  }
  statusMsg->payload[4] &= 0x0F;
  statusMsg->payload[4] |= (sm->getAutoStepSize() & 0xF) << 4;
  statusMsg->payload[5] = (sm->getAutoStepSize() & 0xFF0) >> 4;
  statusMsg->payload[6] = (sm->getAutoDelay() & 0xFF);
  statusMsg->payload[7] = (sm->getAutoDelay() & 0xFF00) >> 8;
  //printMessage(statusMsg);
  writer.writeMessage(statusMsg);
}

