/*
################################################################################
#                                                                              #
#    Ambient Lights Controller Firmware                                        #
#                                                                              #
#    Copyright 2012 Robert Mai                                                 #
#                                                                              #
#    Contact: robert@hsapps.com                                                #
#    URL:     http://www.hsapps.com                                            #
#                                                                              #
#                                                                              #
#    Ambient Lights Controller Firmware is free software: you can              #
#    redistribute it and/or modify it under the terms of the GNU General       #
#    Public License as published by the Free Software Foundation, either       #
#    version 3 of the License, or (at your option) any later version.          #
#                                                                              #
#    Ambient Lights Controller Firmware is distributed in the hope that        #
#    it will be useful, but WITHOUT ANY WARRANTY; without even the implied     #
#    warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.          #
#    See the GNU General Public License for more details.                      #
#                                                                              #
#    You should have received a copy of the GNU General Public License         #
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.     #
#                                                                              #
################################################################################
*/

#ifndef StateSaver_h
#define StateSaver_h

#include "Arduino.h"

#include "LEDController.h"


#define MIN_ADDR 0
#define MAX_ADDR 255

#define TOKEN_START 0x10
#define TOKEN_SKIP 0x11

#define SAVE_INTERVAL_MS 5000

/*
Data will only be stored if RGB stays the same for at least SAVE_INTERVAL_MS milliseconds.

Data will be stored between MIN_ADDR and MAX_ADDR.
data[MIN_ADDR] must be TOKEN_START to indicate that a value has been stored before.
If it's not, nothing will be read and next save will start at MIN_ADDR.

After address MIN_ADDR follow N values with TOKEN_SKIP.
This is followed by one value with TOKEN_START.
This is followed by three single values for R, G and B.

StateSaver will remember the position of the second TOKEN_START.
On next save all data will be stored on address higher and previous TOKEN_START will be set to TOKEN_SKIP.
If new position of second TOKEN_START > MAX_ADDR - 3, it will be set to MIN_ADDR + 1.

*/


class StateSaver
{
  private:
    LEDController* led1;
    int prevH1;
    int prevS1;
    int prevV1;
    unsigned long sameSince1;
    boolean stored1;

    LEDController* led2;
    int prevH2;
    int prevS2;
    int prevV2;
    unsigned long sameSince2;
    boolean stored2;

    int storedH1;
    int storedS1;
    int storedV1;
    int storedH2;
    int storedS2;
    int storedV2;
    
    boolean fading; //are we currently about to fade in?
    unsigned long fadeDelayTarget; //how long shall fading last in ms?
    float fadeDelayTargetF;
    unsigned long fadeStart; //when did we start fading?
    float fadeVTarget1; //target value
    float fadeVTarget2; //target value
    
    int startTokenAddress; //address of current second TOKEN_START
    void writeState(boolean writeChannel1, boolean writeChannel2); //writes current state to EEPROM
    boolean processChannel(LEDController* led, int* prevH, int* prevS, int* prevV, unsigned long* sameSince, boolean* stored);
  public:
    StateSaver();
    void setup(LEDController* vLed1,LEDController* vLed2);
    void process();
    void readState(unsigned long fadeDelay); //reads current state to EEPROM and sets LED colors accordingly, fading in for fadeDelay ms
    void readState(); //reads current state to EEPROM and sets LED colors accordingly and sets fadeDelay to 0ms
};

#endif

