/*
################################################################################
#                                                                              #
#    MyHomeMessageLayer                                                        #
#                                                                              #
#    Copyright 2012 Robert Mai                                                 #
#                                                                              #
#    Contact: robert@hsapps.com                                                #
#    URL:     http://www.hsapps.com                                            #
#                                                                              #
#                                                                              #
#    MyHomeMessageLayer is free software: you can                              #
#    redistribute it and/or modify it under the terms of the GNU General       #
#    Public License as published by the Free Software Foundation, either       #
#    version 3 of the License, or (at your option) any later version.          #
#                                                                              #
#    MyHomeMessageLayer is distributed in the hope that                        #
#    it will be useful, but WITHOUT ANY WARRANTY; without even the implied     #
#    warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.          #
#    See the GNU General Public License for more details.                      #
#                                                                              #
#    You should have received a copy of the GNU General Public License         #
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.     #
#                                                                              #
################################################################################
*/

#include "MyHomeMessageLayer.h"

#include <DebugPrinter.h>

#include "Arduino.h"


#define CTRL_START     0xAA // start byte
#define CTRL_ESC       0xAB // escape byte
#define CTRL_ESC_START 0xAC // start byte replacement
#define CTRL_ESC_ESC   0xAD // escape byte replacement

#define STATE_WAIT_FOR_START           0x00
#define STATE_WAIT_FOR_SERVICE_ID      0x01
#define STATE_WAIT_FOR_TARGET_ID       0x02
#define STATE_WAIT_FOR_SOURCE_ID       0x03
#define STATE_WAIT_FOR_MESSAGE_ID      0x04
#define STATE_WAIT_FOR_DLC_MSB         0x05
#define STATE_WAIT_FOR_DLC_LSB         0x06
#define STATE_WAIT_FOR_HEADER_CRC      0x07
#define STATE_WAIT_FOR_PAYLOAD         0x08
#define STATE_WAIT_FOR_PAYLOAD_CRC_MSB 0x09
#define STATE_WAIT_FOR_PAYLOAD_CRC_LSB 0x0A



extern int __bss_end;
extern void *__brkval;

int get_free_memory()
{
  int free_memory;

  if((int)__brkval == 0)
    free_memory = ((int)&free_memory) - ((int)&__bss_end);
  else
    free_memory = ((int)&free_memory) - ((int)__brkval);

  return free_memory;
}




void mlayer_test() {
}

/**
Initializes a Message struct instance and allocates payload buffer if buffer_size > 0
**/
Message* mlayer_createMessage(int buffer_size) {
  Message* msg = (Message*) malloc(sizeof(Message));
  msg->service_id = 0;
  msg->target_id = 0;
  msg->source_id = 0;
  msg->header_crc = 0;
  msg->payload_length = 0;
  msg->payload_buffer_size = buffer_size;
  if (buffer_size <= 0) {
    msg->payload = NULL;
  }
  else {
    msg->payload = (uint8_t*) malloc(buffer_size*sizeof(uint8_t));
  }
  msg->payload_crc = 0;
  return msg;
}

void mlayer_freeMessage(Message *msg) {
    if (msg != NULL) {
        if (msg->payload != NULL) {
            free(msg->payload);
        }
        free(msg);
    }
}


void MessageReader::init(uint8_t vServiceFilterMask) {
  this->serviceFilterMask = vServiceFilterMask;
  this->state = STATE_WAIT_FOR_START;
  this->msg = NULL;
  this->current_payload_pos = 0;
  this->got_escape = 0;
}

Message* MessageReader::putByte(int c) {
  //check if we received a frame start byte:
  if (c == CTRL_START) {
    //create Message if it does not exist yet:
    if (this->msg == NULL) {
      this->msg = mlayer_createMessage(0);
    }
    this->got_escape = 0;
    this->state = STATE_WAIT_FOR_SERVICE_ID;
    return NULL;
  }
  
  //only process this byte if we are inside a Message frame
  if (this->state != STATE_WAIT_FOR_START) {
    //We're inside a Message frame!
    
    //did we receive an escape byte?
    if (c == CTRL_ESC) {
      this->got_escape = 1;
      return NULL;
    }
    //did we receive an escape byte last time?
    if (this->got_escape == 1) {
      //replace c by original byte
      switch (c) {
        case CTRL_ESC_START:
          c = CTRL_START;
          break;
        case CTRL_ESC_ESC:
          c = CTRL_ESC;
          break;
        default:
          //INVALID!!!
          break;
      }
      this->got_escape = 0;
    }
    
    //store received byte according to current state
    switch (this->state) {
      case STATE_WAIT_FOR_SERVICE_ID:
        //only go on processing this message if it matches the serviceFilterMask:
        if ( (c & this->serviceFilterMask) == 0 )
        {
          //stop processing this message!
          this->state = STATE_WAIT_FOR_START;
          return NULL;
        }
        else
        {
          this->msg->service_id = c;
          this->state = STATE_WAIT_FOR_TARGET_ID;
        }
        break;
      case STATE_WAIT_FOR_TARGET_ID:
        this->msg->target_id = c;
        this->state = STATE_WAIT_FOR_SOURCE_ID;
        break;
      case STATE_WAIT_FOR_SOURCE_ID:
        this->msg->source_id = c;
        this->state = STATE_WAIT_FOR_MESSAGE_ID;
        break;
      case STATE_WAIT_FOR_MESSAGE_ID:
        this->msg->message_id = c;
        this->state = STATE_WAIT_FOR_DLC_MSB;
        break;
      case STATE_WAIT_FOR_DLC_MSB:
        this->msg->payload_length = c <<8;
        this->state = STATE_WAIT_FOR_DLC_LSB;
        break;
      case STATE_WAIT_FOR_DLC_LSB:
        this->msg->payload_length += c;
        this->state = STATE_WAIT_FOR_HEADER_CRC;
        break;
      case STATE_WAIT_FOR_HEADER_CRC:
        this->msg->header_crc = c;
        
        //TODO: check header crc and only continue if it's ok!
        
        //prepare payload reception:
        if (this->msg->payload_length >0) {
          this->state = STATE_WAIT_FOR_PAYLOAD;
          this->current_payload_pos = 0;
          //check if current buffer is large enough for payload:
          if (this->msg->payload_buffer_size < this->msg->payload_length) {
            //not enough, free old buffer, allocate new one
            if (this->msg->payload != NULL) {
              free(this->msg->payload);
            }
            this->msg->payload = (uint8_t*) malloc(this->msg->payload_length*sizeof(uint8_t));
            this->msg->payload_buffer_size = this->msg->payload_length;
          }
        }
        else {
          this->state = STATE_WAIT_FOR_PAYLOAD_CRC_MSB;
        }
        break;
      case STATE_WAIT_FOR_PAYLOAD:
        this->msg->payload[this->current_payload_pos] = c;
        this->current_payload_pos++;
        if (this->current_payload_pos == this->msg->payload_length) {
          //finished receiving payload data
          this->state = STATE_WAIT_FOR_PAYLOAD_CRC_MSB;
        }
        break;
      case STATE_WAIT_FOR_PAYLOAD_CRC_MSB:
        this->msg->payload_crc = c<<8;
        this->state = STATE_WAIT_FOR_PAYLOAD_CRC_LSB;
        break;
      case STATE_WAIT_FOR_PAYLOAD_CRC_LSB:
        this->msg->payload_crc += c;
        this->state = STATE_WAIT_FOR_START;
        //message frame finished, return msg
        return this->msg;
      default:
        break;
    }
  }
  return NULL;
}

Message* MessageReader::takeOverMessage() {
    Message* res = this->msg;
    this->msg = NULL;
    return res;
}


/**
Prints message data to the serial port
**/
void mlayer_printMessage(Message *msg) {
  Serial.println("==============================");
  Serial.println("Message information:");
  if (msg == NULL) {
    Serial.println("  NULL");
    return;
  }
  Serial.print("service_id: 0x");
  Serial.println(msg->service_id, HEX);

  Serial.print("target_id: 0x");
  Serial.println(msg->target_id, HEX);
  
  Serial.print("source_id: 0x");
  Serial.println(msg->source_id, HEX);
  
  Serial.print("message_id: 0x");
  Serial.println(msg->message_id, HEX);
  
  Serial.print("payload_length: ");
  Serial.println(msg->payload_length, DEC);
  
  Serial.print("payload_buffer_size: ");
  Serial.println(msg->payload_buffer_size, DEC);
  
  Serial.print("header_crc: 0x");
  Serial.println(msg->header_crc, HEX);
  
  Serial.print("payload: ");
  for (int i = 0; i<msg->payload_length; i++) {
    Serial.print("0x");
    Serial.print(msg->payload[i],HEX);
    Serial.print(", ");
  }
  Serial.println("");
  
  Serial.print("payload_crc: 0x");
  Serial.println(msg->payload_crc, HEX);
  Serial.println("==============================");
}


void MessageWriter::setPutByteFunction(void (*putByteFunction) (int))
{
    putByte = putByteFunction;
}

void MessageWriter::writeMessage(Message* msg)
{
    //send start byte:
    putByte(CTRL_START);
    
    //send message's data:
    write(msg->service_id);
    write(msg->target_id);
    write(msg->source_id);
    write(msg->message_id);
    write((msg->payload_length & 0xFF00) >> 8); //MSB first
    write(msg->payload_length & 0x00FF);
    write(0); //FIXME: this should be the header CRC!!!
    for (int i=0; i<msg->payload_length; i++) {
        write(msg->payload[i]);
    }
    int crc = 0; //FIXME: this should be the payload CRC!!!
    write((crc & 0xFF00) >> 8); //MSB first
    write(crc & 0x00FF);
}

void MessageWriter::write(int b) //write single byte
{
    switch (b) {
    case CTRL_ESC: //Data contains an escape byte. Escape it.
        putByte(CTRL_ESC);
        putByte(CTRL_ESC_ESC);
        break;
    case CTRL_START: //Data contains a start byte. Escape it.
        putByte(CTRL_ESC);
        putByte(CTRL_ESC_START);
        break;
    default: //No special byte. Just write it.
        putByte(b);
        break;
    }
}