/*
################################################################################
#                                                                              #
#    MyHomeMessageLayer                                                        #
#                                                                              #
#    Copyright 2012 Robert Mai                                                 #
#                                                                              #
#    Contact: robert@hsapps.com                                                #
#    URL:     http://www.hsapps.com                                            #
#                                                                              #
#                                                                              #
#    MyHomeMessageLayer is free software: you can                              #
#    redistribute it and/or modify it under the terms of the GNU General       #
#    Public License as published by the Free Software Foundation, either       #
#    version 3 of the License, or (at your option) any later version.          #
#                                                                              #
#    MyHomeMessageLayer is distributed in the hope that                        #
#    it will be useful, but WITHOUT ANY WARRANTY; without even the implied     #
#    warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.          #
#    See the GNU General Public License for more details.                      #
#                                                                              #
#    You should have received a copy of the GNU General Public License         #
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.     #
#                                                                              #
################################################################################
*/

#ifndef MyHomeMessageLayer_h
#define MyHomeMessageLayer_h

#include <inttypes.h>

//Service ids:
#define SERVICE_NETWORK_MANAGEMENT 0b11111111 //Network management stuff
#define SERVICE_AMBIENT_LIGHTS     0b00000001 //Ambient Lights LED controller
#define SERVICE_WIRELESS_SWITCHES  0b00000010 //Funksteckdosen


void mlayer_test();
int get_free_memory();

struct Message {
  uint8_t service_id;
  uint8_t target_id;
  uint8_t source_id;
  uint8_t message_id;
  uint8_t header_crc;
  uint16_t payload_length;
  uint16_t payload_buffer_size;
  uint8_t* payload;
  uint16_t payload_crc;
};
typedef struct Message Message;

Message* mlayer_createMessage(int buffer_size);
void mlayer_freeMessage(Message *msg);

/**
This class implements a state machine receiving data from a byte stream.
Use putByte() to push all received bytes into the state machine.
**/
class MessageReader
{
  private:
    uint8_t serviceFilterMask;
    int state;
    uint8_t got_escape;
    int current_payload_pos;
    Message* msg;
  public:
    /**
    Initialize the MessageReader instance.
    Must be called once before processing data.
    @param vServiceFilterMask
        only process messages with service_id matching this bitmask.
    **/
    void init(uint8_t vServiceFilterMask);
    /**
    Put all received bytes into the receiver.
    Receiver decodes data.
    If complete message was received, this function returns
    pointer to message data. Data must be processed immediately!
    On next call of putByte, this data could get invalid.
    Do not free this function's resulting data memory!
    **/
    Message* putByte(int c);
    /**
    Take over control of previously received message data for later use.
    Reader will create new one on next reception.
    Caller must free memory of returned data!
    **/
    Message* takeOverMessage();
};

/**
This class implements a writer for messages.
A function must be provided to which the single bytes of the outgoing message will be put.
**/
class MessageWriter
{
    private:
      void (*putByte) (int); //Pointer to a function to put all bytes for the outgoing Message.
    public:
      void setPutByteFunction(void (*putByteFunction) (int));
      void writeMessage(Message* msg);
      void write(int b); //write single byte
};

void mlayer_printMessage(Message *msg);

#endif
