/*
################################################################################
#                                                                              #
#    Ambient Lights Controller Firmware                                        #
#                                                                              #
#    Copyright 2012 Robert Mai                                                 #
#                                                                              #
#    Contact: robert@hsapps.com                                                #
#    URL:     http://www.hsapps.com                                            #
#                                                                              #
#                                                                              #
#    Ambient Lights Controller Firmware is free software: you can              #
#    redistribute it and/or modify it under the terms of the GNU General       #
#    Public License as published by the Free Software Foundation, either       #
#    version 3 of the License, or (at your option) any later version.          #
#                                                                              #
#    Ambient Lights Controller Firmware is distributed in the hope that        #
#    it will be useful, but WITHOUT ANY WARRANTY; without even the implied     #
#    warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.          #
#    See the GNU General Public License for more details.                      #
#                                                                              #
#    You should have received a copy of the GNU General Public License         #
#    along with this program.  If not, see <http://www.gnu.org/licenses/>.     #
#                                                                              #
################################################################################
*/

#include "SerialInterface.h"
#include "WProgram.h"


SerialInterface::SerialInterface()
{
}

void SerialInterface::setup(int baudRate, StateMachine* vSm, LEDController* vLed)
{
    sm = vSm;
    led = vLed;
    VERBOSE = 0;
    Serial.begin(baudRate);
}

/*
   Receives up to three bytes via Serial and interprets String as decimal integer.
   If char is received which is not a number -1 will be returned. Reception will be
*/
int SerialInterface::receiveIntValue()
{
  int res = 0;
  int msg = 0;
  int i = 100;
  while (true)
  {
    while(!Serial.available())
    {
    }
    msg = Serial.read();
    if ((msg<'0') || (msg>'9') )
    {
      if (VERBOSE)
      {
        Serial.print("This byte is not a number: ");
        Serial.println(msg,DEC);
      }
      return -1;
    }
    res += (msg-'0')*i;
    if (i==1)
      break;
    i/=10;
  }
  return res;    
}

void SerialInterface::process()
{
  byte msg = 0;
  if (Serial.available()) {
    msg = Serial.read();
    if (msg=='a') //activate auto mode
    {
      if (VERBOSE)
      {
        Serial.println("Activating auto mode.");
      }
      sm->onSetAuto(true);
    }
    else if (msg=='h') //deactivate auto mode (hold color).
    {
      if (VERBOSE)
      {
        Serial.println("Holding this color.");
      }
      sm->onSetAuto(false);
    }
    else if (msg=='c') //set color (RGB)
    {
      int tmp = 0;
      int tmpR = 0;
      int tmpG = 0;
      int tmpB = 0;
  
      if (VERBOSE)
        Serial.println("Waiting for red value.");
      tmpR = receiveIntValue();
      if (tmpR == -1)
        return;
      if (VERBOSE)
      {
        Serial.println(tmpR,DEC);
        Serial.println("Waiting for green value.");
      }
      tmpG = receiveIntValue();
      if (tmpG == -1)
        return;
      if (VERBOSE)
      {
        Serial.println(tmpG,DEC);
       Serial.println("Waiting for blue value.");
      }
      tmpB = receiveIntValue();
      if (tmpB == -1)
        return;
      if (VERBOSE)
        Serial.println(tmpB,DEC);
      sm->onSetRGB(tmpR,tmpG,tmpB);
    }
    else if (msg=='d') //set color (HSV)
    {
      int tmp = 0;
      int tmpH = 0;
      int tmpS = 0;
      int tmpV = 0;
  
      if (VERBOSE)
        Serial.println("Waiting for Hue value.");
      tmpH = receiveIntValue();
      if (tmpH == -1)
        return;
      if (VERBOSE)
      {
        Serial.println(tmpH,DEC);
        Serial.println("Waiting for Saturation value.");
      }
      tmpS = receiveIntValue();
      if (tmpS == -1)
        return;
      if (VERBOSE)
      {
        Serial.println(tmpS,DEC);
        Serial.println("Waiting for brightness value.");
      }
      tmpV = receiveIntValue();
      if (tmpV == -1)
        return;
      if (VERBOSE)
        Serial.println(tmpV,DEC);
      sm->onSetHSV(tmpH,tmpS,tmpV);
    }
    else if (msg=='s') //set auto mode speed
    {
      int stepSize = 0;
      int stepDelay = 0;
      if (VERBOSE)
        Serial.println("Waiting for stepSize.");
      stepSize = receiveIntValue();
      if (stepSize == -1)
        return;
      if (VERBOSE)
      {
        Serial.println(stepSize,DEC);
        Serial.println("Waiting for stepDelay.");
      }
      stepDelay = receiveIntValue();
      if (stepDelay == -1)
        return;
      if (VERBOSE)
        Serial.println(stepDelay,DEC);
      sm->onSetAutoValues(stepSize,stepDelay);
    }
    else if (msg=='v') //activate/deactivate verbose mode
    {
      int tmp = 0;
      if (VERBOSE)
        Serial.println("Waiting for verbose value.");
      tmp = receiveIntValue();
      if ( (tmp < 0) || (tmp > 1) )
        return;
      VERBOSE = tmp;
      if (VERBOSE)
        Serial.println(VERBOSE,DEC);
    }
    else if (msg=='r') //read current state
    {
      //Send current state machine state and current HSV-values:
      Serial.print("status:");
      Serial.print(SERIAL_INTERFACE_VERSION,DEC); //SerialInterface version number
      Serial.print("."); //divider
      Serial.print(sm->getState(),DEC); //State as single digit decimal
      Serial.print("."); //divider
      Serial.print(led->getHue(),DEC);
      Serial.print("."); //divider
      Serial.print(led->getSaturation(),DEC);
      Serial.print("."); //divider
      Serial.print(led->getValue(),DEC);
      Serial.print("."); //divider
      Serial.print(sm->getAutoStepSize(),DEC);
      Serial.print("."); //divider
      Serial.println(sm->getAutoDelay(),DEC);
    }
    else
    {
      if (VERBOSE)
      {
        Serial.print("Received unknown command intro: ");
        Serial.println(msg,DEC);
      }
    }
  }
}

